<?php
session_start();
header('Content-Type: application/json');
require_once '../config/db.php';
$conn = getDB();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (!isset($_SESSION['user_id'])) {
    error_log("Application API: Unauthorized access - user_id not set in session.");
    echo json_encode(['success' => false, 'message' => 'Unauthorized access. Please log in.']);
    exit;
}

// Verify payment exists before allowing application
// In api/application.php, replace the payment check and duplicate check with:

// 1. First verify payment exists and get application_id
$payment_check = $conn->prepare("
    SELECT application_id FROM payments 
    WHERE user_id = ? AND status = 'successful'
    ORDER BY created_at DESC LIMIT 1
");
$payment_check->bind_param("i", $_SESSION['user_id']);
$payment_check->execute();
$payment_result = $payment_check->get_result();

if ($payment_result->num_rows === 0) {
    echo json_encode(['success' => false, 'message' => 'Payment required before application.']);
    exit;
}

$payment_data = $payment_result->fetch_assoc();
$application_id = $payment_data['application_id'];

// 2. Check if application ALREADY EXISTS for this application_id
$app_check = $conn->prepare("SELECT id FROM applications WHERE application_id = ?");
$app_check->bind_param("s", $application_id);
$app_check->execute();

if ($app_check->get_result()->num_rows > 0) {
    echo json_encode(['success' => false, 'message' => 'Application already submitted for this payment.']);
    exit;
}

// 3. Continue with application submission if no duplicate found in applications table



// Continue with the application submission if no duplicate found

// Collect and sanitize POST data
$user_id = $_SESSION['user_id'];
$section = isset($_POST['applicantType']) ? trim($_POST['applicantType']) : '';
$full_name = isset($_POST['fullName']) ? trim($_POST['fullName']) : '';
$gender = isset($_POST['gender']) ? trim($_POST['gender']) : '';

// With proper date validation:
$dob = isset($_POST['dob']) ? trim($_POST['dob']) : '';
if (!empty($dob)) {
    try {
        // Validate the date format
        $date = DateTime::createFromFormat('Y-m-d', $dob);
        if (!$date || $date->format('Y-m-d') !== $dob) {
            throw new Exception("Invalid date format");
        }
        // Format for MySQL
        $dob = $date->format('Y-m-d');
    } catch (Exception $e) {
        error_log("Invalid DOB format: " . $dob);
        echo json_encode(['success' => false, 'message' => 'Invalid date of birth format. Use YYYY-MM-DD']);
        exit;
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Date of birth is required']);
    exit;
}
$lga = isset($_POST['lga']) ? trim($_POST['lga']) : '';
$state = isset($_POST['state']) ? trim($_POST['state']) : '';

$parent_name = isset($_POST['parentName']) ? trim($_POST['parentName']) : '';
$parent_occupation = isset($_POST['parentOccupation']) ? trim($_POST['parentOccupation']) : '';
$parent_phone = isset($_POST['parentPhone']) ? trim($_POST['parentPhone']) : '';
$parent_email = isset($_POST['parentEmail']) ? trim($_POST['parentEmail']) : '';

$current_school = isset($_POST['currentSchool']) ? trim($_POST['currentSchool']) : null;
$current_class = isset($_POST['currentClass']) ? trim($_POST['currentClass']) : null;
$application_date = isset($_POST['applicationDate']) ? trim($_POST['applicationDate']) : date('Y-m-d');

$health_status = isset($_POST['healthStatus']) ? trim($_POST['healthStatus']) : '';
$disability = isset($_POST['disability']) ? trim($_POST['disability']) : '';
$other_disability = ($disability === 'OTHER' && isset($_POST['otherDisability'])) ? trim($_POST['otherDisability']) : null;
$declaration = isset($_POST['declaration']) ? 1 : 0;

$age = null;
if ($dob) {
    try {
        $dobDate = new DateTime($dob);
        $now = new DateTime();
        $age = $now->diff($dobDate)->y;
    } catch (Exception $e) {
        error_log("Application API: DOB parsing error: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Invalid Date of Birth format.']);
        exit;
    }
}

$required_fields_missing = false;
$required_fields = [
    $section, $full_name, $gender, $dob, $lga, $state,
    $parent_name, $parent_occupation, $parent_phone, $parent_email,
    $health_status, $disability, $declaration
];

foreach ($required_fields as $field) {
    if (empty($field) && $field !== 0) {
        $required_fields_missing = true;
        break;
    }
}

if ($section === 'junior') {
    if (empty($current_school) || empty($current_class)) {
        $required_fields_missing = true;
    }
}

if ($disability === 'OTHER' && empty($other_disability)) {
    $required_fields_missing = true;
}

if ($required_fields_missing) {
    error_log("Application API: Missing required fields.");
    echo json_encode(['success' => false, 'message' => 'All required fields must be filled.']);
    exit;
}

if ($section === 'primary' && ($age === null || $age < 5)) {
    error_log("Application API: Primary applicant age validation failed (Age: " . ($age ?? 'null') . ")");
    echo json_encode(['success' => false, 'message' => 'Primary applicants must be at least 5 years old.']);
    exit;
} elseif ($section === 'junior' && ($age === null || $age < 12)) {
    error_log("Application API: Junior applicant age validation failed (Age: " . ($age ?? 'null') . ")");
    echo json_encode(['success' => false, 'message' => 'Junior applicants must be at least 12 years old.']);
    exit;
}

// File upload handling
function handle_upload($field_name, $prefix, $application_id, $required = false) {
    if (isset($_FILES[$field_name]) && $_FILES[$field_name]['error'] == 0) {
        $target_dir = '../uploads/';
        if (!is_dir($target_dir)) {
            mkdir($target_dir, 0777, true);
        }
        $ext = pathinfo($_FILES[$field_name]['name'], PATHINFO_EXTENSION);
        $filename = $prefix . '_' . $application_id . '.' . $ext;
        $target_file = $target_dir . $filename;

        $allowed_types = ['pdf', 'jpg', 'jpeg', 'png'];
        if (!in_array(strtolower($ext), $allowed_types)) {
            return ['error' => 'Invalid file type for ' . $field_name . '. Only PDF, JPG, JPEG, PNG allowed.'];
        }

        if (move_uploaded_file($_FILES[$field_name]['tmp_name'], $target_file)) {
            return ['filename' => $filename];
        } else {
            return ['error' => 'Failed to move uploaded file for ' . $field_name . '.'];
        }
    } elseif ($required) {
        return ['error' => 'Required file ' . $field_name . ' is missing.'];
    }
    return ['filename' => null];
}

$file_fields = [
    'birthCert' => ['prefix' => 'birthcert', 'required' => true],
    'lgaCert' => ['prefix' => 'lgacert', 'required' => true],
    'medicalReport' => ['prefix' => 'medicalreport', 'required' => true],
    'immunizationCard' => ['prefix' => 'immunizationcard', 'required' => true],
    'photograph' => ['prefix' => 'photograph', 'required' => true],
];

if ($section === 'junior') {
    $file_fields['primaryCert'] = ['prefix' => 'primarycert', 'required' => true];
} else {
    $primary_cert = null;
}

$uploaded_files = [];
foreach ($file_fields as $field_name => $props) {
    $upload_result = handle_upload($field_name, $props['prefix'], $application_id, $props['required']);
    if (isset($upload_result['error'])) {
        error_log("Application API: File upload error for " . $field_name . ": " . $upload_result['error']);
        echo json_encode(['success' => false, 'message' => $upload_result['error']]);
        exit;
    }
    $uploaded_files[$field_name] = $upload_result['filename'];
}

$birth_cert = $uploaded_files['birthCert'] ?? null;
$lga_cert = $uploaded_files['lgaCert'] ?? null;
$medical_report = $uploaded_files['medicalReport'] ?? null;
$immunization_card = $uploaded_files['immunizationCard'] ?? null;
$primary_cert = $uploaded_files['primaryCert'] ?? null;
$photograph = $uploaded_files['photograph'] ?? null;


error_log("Application API: Attempting to INSERT application with ID: " . $application_id);

// Insert into applications table
$sql = "INSERT INTO applications (
    user_id, application_id, section, full_name, gender, dob, age, lga, state, parent_name, 
    parent_occupation, parent_phone, parent_email, current_school, current_class, 
    application_date, health_status, disability, other_disability, birth_cert, 
    lga_cert, medical_report, immunization_card, primary_cert, photograph, declaration
) VALUES (?, ?, ?, ?, ?, STR_TO_DATE(?, '%Y-%m-%d'), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

$stmt = $conn->prepare($sql);
if (!$stmt) {
    error_log("Application API: Prepare failed: (" . $conn->errno . ") " . $conn->error);
    echo json_encode(['success' => false, 'message' => 'Database prepare failed: ' . $conn->error]);
    exit;
}

// Change the type string to use 'i' for age (7th parameter)
$stmt->bind_param(
    "isssssi" . str_repeat("s", 18) . "i", // Notice the 'i' for age
    $user_id, 
    $application_id, 
    $section, 
    $full_name, 
    $gender, 
    $dob,
    $age, // Now binding as integer
    $lga, 
    $state, 
    $parent_name, 
    $parent_occupation,
    $parent_phone, 
    $parent_email, 
    $current_school, 
    $current_class,
    $application_date, 
    $health_status, 
    $disability, 
    $other_disability,
    $birth_cert, 
    $lga_cert, 
    $medical_report, 
    $immunization_card,
    $primary_cert, 
    $photograph, 
    $declaration
);

if ($stmt->execute()) {
    error_log("Application API: Application INSERT successful for ID: " . $application_id);

    // Update payment record with application completion status
    $update_payment = $conn->prepare("
        UPDATE payments SET application_completed = 1 
        WHERE application_id = ? AND user_id = ?
    ");
    if (!$update_payment) {
        error_log("Application API: Update payment prepare failed: " . $conn->error);
    } else {
        $update_payment->bind_param("si", $application_id, $user_id);
        if (!$update_payment->execute()) {
            error_log("Application API: Failed to UPDATE payment status for ID: " . $application_id . " Error: " . $update_payment->error);
        } else {
            error_log("Application API: Payment status UPDATED for ID: " . $application_id);
        }
        $update_payment->close();
    }
    
    // Insert into admission_status table
    $sql2 = "INSERT INTO admission_status (application_id, status) VALUES (?, 'pending')";
    $stmt2 = $conn->prepare($sql2);
    if (!$stmt2) {
        error_log("Application API: Admission status prepare failed: (" . $conn->errno . ") " . $conn->error);
    } else {
        $stmt2->bind_param("s", $application_id);
        if (!$stmt2->execute()) {
            error_log("Application API: Failed to INSERT admission status for ID: " . $application_id . " Error: " . $stmt2->error);
        } else {
            error_log("Application API: Admission status INSERT successful for ID: " . $application_id);
        }
        $stmt2->close();
    }
    
    echo json_encode(['success' => true, 'application_id' => $application_id]);
} else {
    error_log("Application API: Application submission FAILED for ID: " . $application_id . " Error: " . $stmt->error);
    echo json_encode(['success' => false, 'message' => 'Application submission failed: ' . $stmt->error]);
}

$stmt->close();
$conn->close();
?>