<?php
// api/generate_admission_letter.php
session_start();
require_once '../config/db.php';
require_once('../assets/fpdf/fpdf.php');

if (!isset($_SESSION['user_id']) || !isset($_GET['application_id'])) {
    die('Unauthorized or missing application ID.');
}

$conn = getDB();
$user_id = $_SESSION['user_id'];
$application_id = $_GET['application_id'];

// Get application data
$sql = "SELECT a.*, s.status, s.score 
        FROM applications a 
        LEFT JOIN admission_status s ON a.application_id = s.application_id 
        WHERE a.user_id = ? AND a.application_id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("is", $user_id, $application_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    die('Application not found.');
}

$app = $result->fetch_assoc();
$stmt->close();

// Only allow download if status is 'accepted'
if ($app['status'] !== 'accepted') {
    die('Admission letter not available for your application status.');
}

// Enhanced but compatible PDF Class
class EnhancedAdmissionLetterPDF extends FPDF {
    
    function Header() {
        // Header background - reduced height
        $this->SetFillColor(52, 73, 85); // Dark blue
        $this->Rect(0, 0, 210, 30, 'F');
        
        // School logo (if exists, otherwise skip)
        if (file_exists('../assets/images/logo.png')) {
            $this->Image('../assets/images/logo.png', 15, 5, 20);
        }
        
        // School name and details in white - more compact
        $this->SetTextColor(255, 255, 255);
        $this->SetFont('Arial', 'B', 14);
        $this->SetXY(40, 6);
        $this->Cell(0, 6, 'UBEC MODEL SMART SCHOOL DAURA', 0, 1, 'L');
        
        $this->SetFont('Arial', '', 9);
        $this->SetXY(40, 14);
        $this->Cell(0, 5, 'Excellence in Education - Character Development - Innovation', 0, 1, 'L');
        
        $this->SetFont('Arial', 'I', 9);
        $this->SetXY(40, 21);
        $this->Cell(0, 5, '2024/2025 Academic Session', 0, 1, 'L');
        
        // Gold decorative line
        $this->SetDrawColor(255, 193, 7);
        $this->SetLineWidth(1);
        $this->Line(10, 28, 200, 28);
        
        $this->Ln(12);
    }
    
    function Footer() {
        $this->SetY(-15);
        
        // Footer line
        $this->SetDrawColor(41, 84, 142);
        $this->SetLineWidth(0.5);
        $this->Line(10, $this->GetY(), 200, $this->GetY());
        
        $this->SetTextColor(100, 100, 100);
        $this->SetFont('Arial', '', 7);
        $this->Ln(2);
        
        // Contact information - more compact
        $this->Cell(0, 3, 'Daura, Katsina State | Phone: +234-XXX-XXX-XXXX | Email: info@ubecmodelschool.edu.ng | www.ubecmodelschool.edu.ng', 0, 1, 'C');
    }
    
    // Helper function for styled boxes
    function StyledBox($x, $y, $w, $h, $text, $bgColor = array(248, 249, 250)) {
        $this->SetFillColor($bgColor[0], $bgColor[1], $bgColor[2]);
        $this->SetDrawColor(200, 200, 200);
        $this->Rect($x, $y, $w, $h, 'DF');
        $this->SetXY($x + 2, $y + 2);
        $this->MultiCell($w - 4, 5, $text, 0, 'L');
    }
    
    // Section header with decorative line - more compact
    function SectionHeader($title) {
        $this->Ln(3);
        $this->SetFont('Arial', 'B', 11);
        $this->SetTextColor(41, 84, 142);
        $this->Cell(0, 6, $title, 0, 1, 'L');
        
        // Decorative underline
        $this->SetDrawColor(255, 193, 7);
        $this->SetLineWidth(0.8);
        $this->Line(10, $this->GetY() - 1, 10 + $this->GetStringWidth($title), $this->GetY() - 1);
        $this->Ln(2);
    }
}

// Create PDF with tighter margins
$pdf = new EnhancedAdmissionLetterPDF();
$pdf->SetAutoPageBreak(true, 20);
$pdf->AddPage();

// Reset text color
$pdf->SetTextColor(51, 51, 51);

// Date and Reference - more compact
$pdf->SetFont('Arial', '', 9);
$pdf->Cell(95, 5, 'Date: ' . date('F j, Y'), 0, 0, 'L');
$pdf->Cell(95, 5, 'Ref: UMS/ADM/' . date('Y') . '/' . str_pad($application_id, 4, '0', STR_PAD_LEFT), 0, 1, 'R');
$pdf->Ln(6);

// Main Title - smaller
$pdf->SetFont('Arial', 'B', 16);
$pdf->SetTextColor(41, 84, 142);
$pdf->Cell(0, 8, 'LETTER OF ADMISSION', 0, 1, 'C');

// Decorative line under title
$pdf->SetDrawColor(255, 193, 7);
$pdf->SetLineWidth(1.5);
$pdf->Line(65, $pdf->GetY() - 1, 145, $pdf->GetY() - 1);
$pdf->Ln(8);

// Recipient section - more compact
$pdf->SetTextColor(51, 51, 51);
$pdf->SetFont('Arial', 'B', 10);
$pdf->Cell(0, 5, 'TO:', 0, 1, 'L');
$pdf->Ln(1);

// Recipient info in styled box - smaller
$recipientText = $app['parent_name'] . "\n" . $app['parent_email'];
$pdf->StyledBox(10, $pdf->GetY(), 100, 16, $recipientText);
$pdf->Ln(18);

// Subject - smaller
$pdf->SetFont('Arial', 'B', 10);
$pdf->SetTextColor(41, 84, 142);
$pdf->Cell(0, 6, 'SUBJECT: ADMISSION OFFER', 0, 1, 'L');
$pdf->Ln(3);

// Main content - tighter spacing
$pdf->SetFont('Arial', '', 10);
$pdf->SetTextColor(51, 51, 51);

$greeting = 'Dear ' . $app['parent_name'] . ',';
$pdf->MultiCell(0, 5, $greeting, 0, 'L');
$pdf->Ln(3);

$mainText = 'We are pleased to inform you that ' . $app['full_name'] . 
           ' has been offered admission into our ' . ucfirst($app['section']) . 
           ' Section for the 2024/2025 academic session.';
$pdf->MultiCell(0, 5, $mainText, 0, 'J');
$pdf->Ln(5);

// Admission Details Section
$pdf->SectionHeader('ADMISSION DETAILS');

// Details table with tighter formatting
$pdf->SetFont('Arial', '', 9);
$details = array(
    array('Application ID:', $application_id),
    array('Student Name:', $app['full_name']),
    array('Section:', ucfirst($app['section'])),
    array('Admission Score:', ($app['score'] ?? 'N/A') . '%'),
    array('Session:', '2024/2025')
);

foreach ($details as $i => $detail) {
    $fill = ($i % 2 == 0);
    if ($fill) {
        $pdf->SetFillColor(248, 249, 250);
    } else {
        $pdf->SetFillColor(255, 255, 255);
    }
    
    $pdf->SetFont('Arial', 'B', 9);
    $pdf->Cell(45, 5, $detail[0], 1, 0, 'L', true);
    $pdf->SetFont('Arial', '', 9);
    $pdf->Cell(145, 5, $detail[1], 1, 1, 'L', true);
}
$pdf->Ln(5);

// Requirements Section
$pdf->SectionHeader('REGISTRATION REQUIREMENTS');

$requirements = 'Please complete the registration process by paying the required fees and submitting all necessary documents by ' . 
               date('F j, Y', strtotime('+30 days')) . '.';
$pdf->SetFont('Arial', '', 10);
$pdf->MultiCell(0, 5, $requirements, 0, 'J');
$pdf->Ln(5);

// Important Notice Box - smaller
$pdf->SetDrawColor(255, 193, 7);
$pdf->SetFillColor(255, 248, 225);
$pdf->SetLineWidth(0.8);
$noticeText = 'IMPORTANT: This admission offer is valid for 30 days from the date of this letter.';
$pdf->Rect(10, $pdf->GetY(), 190, 12, 'DF');
$pdf->SetXY(12, $pdf->GetY() + 3);
$pdf->SetFont('Arial', 'B', 9);
$pdf->MultiCell(186, 5, $noticeText, 0, 'C');
$pdf->Ln(15);

// Congratulations - more compact
$pdf->SetFont('Arial', '', 10);
$congratsText = 'Congratulations and we look forward to welcoming your ward to our school community.';
$pdf->MultiCell(0, 5, $congratsText, 0, 'J');
$pdf->Ln(8);

// Closing
$pdf->Cell(0, 5, 'Sincerely,', 0, 1, 'L');
$pdf->Ln(10);

// Signature section - more compact
$pdf->SetDrawColor(51, 51, 51);
$pdf->Line(10, $pdf->GetY(), 60, $pdf->GetY());
$pdf->Ln(3);
$pdf->SetFont('Arial', 'B', 10);
$pdf->Cell(0, 5, 'Dr. Amina Mohammed', 0, 1, 'L');
$pdf->SetFont('Arial', '', 9);
$pdf->Cell(0, 4, 'Principal', 0, 1, 'L');
$pdf->Cell(0, 4, 'UBEC Model Smart School Daura', 0, 1, 'L');

// Official stamp area - smaller
$pdf->SetXY(130, $pdf->GetY() - 15);
$pdf->SetDrawColor(41, 84, 142);
$pdf->Rect(130, $pdf->GetY(), 45, 15, 'D');
$pdf->SetXY(133, $pdf->GetY() + 6);
$pdf->SetFont('Arial', 'I', 7);
$pdf->Cell(39, 3, '[OFFICIAL STAMP]', 0, 0, 'C');

// Output PDF
$filename = 'Admission_Letter_' . $application_id . '_' . date('Y') . '.pdf';
$pdf->Output('D', $filename);
exit;
?>